<?php
/**
 * Plugin Name: CrawlConsole Analytics
 * Plugin URI: https://crawlconsole.com
 * Description: Enhance your WordPress website's visibility in LLMs with AI analytics by CrawlConsole.
 * Version: 1.0.4
 * Author: CrawlConsole
 * Author URI: https://crawlconsole.com
 * Text Domain: crawlconsole-analytics
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

define('CRAWLCONSOLE_ANALYTICS_VERSION', '1.0.4');
define('CRAWLCONSOLE_ANALYTICS_DIR', plugin_dir_path(__FILE__));
define('CRAWLCONSOLE_ANALYTICS_URL', plugin_dir_url(__FILE__));

require_once CRAWLCONSOLE_ANALYTICS_DIR . 'includes/class-request-middleware.php';
require_once CRAWLCONSOLE_ANALYTICS_DIR . 'includes/class-admin.php';
require_once CRAWLCONSOLE_ANALYTICS_DIR . 'includes/class-api.php';


function crawlconsole_analytics_init() {
    $admin = new CrawlConsole_Analytics_Admin();
    $admin->init();
    
    $middleware = new CrawlConsole_Analytics_Request_Middleware();
    $middleware->init();
}

add_action('init', 'crawlconsole_analytics_init');

/**
 * Enqueue the CrawlConsole tracker script on the frontend when a project key exists
 */
function crawlconsole_analytics_enqueue_tracker() {
    if (is_admin()) {
        return;
    }

    $project_key = get_option('crawlconsole_analytics_project_key', '');
    if (empty($project_key)) {
        return;
    }

    $handle = 'crawlconsole-analytics-tracker';
    $src    = 'https://analytics.crawlconsole.com/tracker.js';

    // Enqueue the script
    wp_enqueue_script(
        $handle,
        $src,
        array(),
        CRAWLCONSOLE_ANALYTICS_VERSION,
        true
    );

    // Mark as defer (supported by WordPress core)
    if (function_exists('wp_script_add_data')) {
        wp_script_add_data($handle, 'defer', true);
    }

    // Inject data-project-key attribute (and ensure defer) for this handle only
    add_filter(
        'script_loader_tag',
        function ($tag, $handle_param, $src_param) use ($handle, $project_key) {
            if ($handle_param !== $handle) {
                return $tag;
            }
            return sprintf(
                '<script src="%s" data-project-key="%s" defer></script>',
                esc_url($src_param),
                esc_attr($project_key)
            );
        },
        10,
        3
    );
}

add_action('wp_enqueue_scripts', 'crawlconsole_analytics_enqueue_tracker');

function crawlconsole_analytics_activate() {
    add_option('crawlconsole_analytics_api_key', '');
    add_option('crawlconsole_analytics_enabled', true);
}

function crawlconsole_analytics_deactivate() {
    delete_option('crawlconsole_analytics_api_key');
    delete_option('crawlconsole_analytics_enabled');
}

register_activation_hook(__FILE__, 'crawlconsole_analytics_activate');
register_deactivation_hook(__FILE__, 'crawlconsole_analytics_deactivate');
